#ifndef KMS_MODEL_DEVICE_LIST_H_
#define KMS_MODEL_DEVICE_LIST_H_

#include <stdbool.h>

#include "global_types.h"

#include "model/credentials.h"

typedef enum device_id_type_t
{
	DEVNODE_PATH=0,
	SYSFS_PATH
} device_id_type_t;

typedef struct kms_device_t kms_device_t;

typedef struct kms_device_t
{
	//---- RIGHTS MGNT ----------------

	bool apply_credentials;

	kms_credentials_t credentials;

	//---- DEVICE LIST PTR ------------

	kms_device_t *next_in_list_ptr;
	kms_device_t *previous_in_list_ptr;

	//---- DEVICE IDENTIFIER ----------

	device_id_type_t id_type;

	char identifier[];
} kms_device_t;

/**
 * Called to add a device to the list of devices we have to wait for.
 *
 * Use this function if you do not plan to set the access mode and the user and group ids of the corresponding devnode
 * of the device.
 *
 * \param identifier the identifier of the device which is either a device node or a syspath
 * \return returns RESULT_OK in case of no ERROR
 * 		- RESULT_INVALID is returned in case the identifier is neither an absolute devnode path or an absolute syspath
 * 		- RESULT_NORESOURCES is returned in case of memory allocation issues
 */
error_code_t device_list_add_no_credentials(const char *identifier);

/**
 * Called to add a device to the list of devices we have to wait for.
 *
 * Use this function if you additionally want to set user and group ids as well as the access_mode of the
 * corresponding device node once it appears.
 *
 * \param identifier the identifier of the device which is either a device node or a syspath
 * \param credentials the credentials to set
 * \return returns RESULT_OK in case of no ERROR
 * 		- RESULT_INVALID is returned in case the identifier is neither an absolute devnode path or an absolute syspath
 * 		- RESULT_NORESOURCES is returned in case of memory allocation issues
 */
error_code_t device_list_add(const char *identifier, const kms_credentials_t *credentials);

/**
 * returns the device that fits to the given sysfs_path or devnode_path.
 * Devices have been created either with the sysfs_path or with the devnode_path as identifier. Depending on the
 * identifier used to create the device, either sysfs_path or devnode_path is used to identify the device.
 * \param sysfs_path the sysfs path of the searched device if set when the device was created
 * \param devnode_path the devnode path of the searched device if set when the device was created
 * \return a pointer to the device data structure if the device could be found. NULL if the device is not in the list.
 */
kms_device_t *device_list_get_device(const char *sysfs_path, const char *devnode_path);

/**
 * This function removes the given device from the list of devices to wait for and frees the memory allocated by
 * the data structure representing the device.
 * \param the device structure of the device to remove. Be careful, the pointer becomes invalid after this call.
 */
void device_list_remove_and_free(kms_device_t *device);

/**
 * returns the first device in the list or NULL in case the list is empty
 * \return the first device in the list or NULL in case the list is empty
 */
kms_device_t *device_list_get_first_device(void);

/**
 * returns the device in the list located after the given one.
 * NULL is returned in case given one is NULL or the given one is the last one in the list.
 * \param device the previous device of the requested one
 * \return the device next to the given one or NULl
 */
kms_device_t *device_list_get_next_device(kms_device_t *device);

/**
 * returns true in case the list of devices is empty.
 * \return true in case the list is empty at the time of the call
 */
bool device_list_is_empty(void);

/**
 * call to empty the list
 */
void device_list_clear(void);

#endif

